// ==UserScript==
// @name         銀行明細ページ - 文字列置換 + 不要ボタン非表示
// @namespace    http://tampermonkey.net/
// @version      0.2
// @description  明細ページの特定文字列を置換し、印刷・CSV・口座連絡書ボタンを非表示にする
// @author       You
// @match        https://*.bank.co.jp/*      // ← 実際の銀行のURLに合わせて修正してください
// @match        https://ib.*.bank.co.jp/*   // 例：ネットバンキングのドメイン
// @grant        none
// ==/UserScript==

(function () {
    'use strict';

    // ====================
    //  1. 文字列置換の設定
    // ====================
    const replacements = [
    {
        from: "XXD",
        to:   "fffa"
    },
    {
        from: "xxsf",
        to:   "fffa"
    },
    {
        from: "振込サービス　SMBCM",
        to:   "パソコン振込 ｵｵｻｷ ｹｲｺ"
    },
    {
        from: "SMBCM",
        to:   "パソコン振込 ｵｵｻｷ ｹｲｺ"
    },
    {
        from: "振込　ACｻ-ﾋﾞｽｾﾝﾀ-",
        to:   "パソコン振込 ｵｵｻｷ ｹｲｺ"
    },
    {
        from: "SMBC(ACｻ-ﾋﾞｽ",
        to:   "パソコン振込 ｵｵｻｷ ｹｲｺ"
    },
    {
        from: "V000000",
        to:   "ﾐﾂｲｽﾐﾄﾓ　ﾍﾝｷﾝ"
    },
    {
        from: "パソコン振込 ﾐﾂｷﾞ ｱﾔﾅ",
        to:   "パソコン振込 ｵｵｻｷ ｹｲｺ"
    },
    {
        from: "V614212　日本信用情報機構",
        to:   "カード　ｾﾌﾞﾝBK0UIN"
    },
    {
        from: "振込　ｻｸﾏ ﾕｳｽｹ",
        to:   "カード　ｾﾌﾞﾝBK0UIN"
    },
    {
        from: "パソコン振込 ﾖｼｶﾜ ﾋｻﾄｷ",
        to:   "カード　ｾﾌﾞﾝBK0UIN"
    },
    {
        from: "ＤＭＭ．ｃｏｍ",
        to:   "ﾌｧﾐﾘｰﾏｰﾄ"
    },
    {
        from: "ＰＡＹＰＡＬ　＊ＡＢＨＩＮＡＶＬＹＴ",
        to:   "ＯＰＥＮＡＩ"
    },
    {
        from: "Ａｍａｚｏｎ　Ｄｏｗｎｌｏａｄｓ",
        to:   "ﾌｧﾐﾘｰﾏｰﾄ"
    },
    {
        from: "Ｂｉｌｌ．Ｋｉｎｇｓｕｍｍｉｔ．Ｃｏｍ",
        to:   "ＯＰＥＮＡＩ"
    },
    {
        from: "振込　ｿｳﾏ ﾅｵﾄ",
        to:   "パソコン振込 ｵｵｻｷ ｹｲｺ"
    },
    {
        from: "CT ｶﾄﾞﾜｷ ﾔｽﾉﾘ",
        to:   "パソコン振込 ｵｵｻｷ ｹｲｺ"
    },
    {
        from: "ＰＡＹＰＡＬ　＊ＫＩＮＧＵＩＮＤＩＧＩ",
        to:   "ＯＰＥＮＡＩ"
    },
    {
        from: "Ｐａｔｒｅｏｎ＊　Ｍｅｍｂｅｒｓｈｉｐ",
        to:   "ﾌｧﾐﾘｰﾏｰﾄ"
    },
    {
        from: "ｖａｒｒｏｃｋｇｒｏｕｐ．ｃｏｍ",
        to:   "さくらインターネット"
    },
    {
        from: "ＡＭＺ＊ＮＯＴＥ",
        to:   "ﾌｧﾐﾘｰﾏｰﾄ"
    },
    {
        from: "PAYPALWALLET",
        to:   "さくらインターネット"
    },
    {
        from: "パソコン振込 ﾍﾞﾝ)ｱｲﾃｲｼﾞｴｲﾎｳﾘﾂｼﾞﾑｼﾖｱｽﾞｶﾘｸﾞﾁ",
        to:   "パソコン振込 ｵｵｻｷ ｹｲｺ"
    },
    {
        from: "パソコン振込 ﾍﾞﾝ)ｲﾁｶﾞﾔﾁﾕｳｵｳﾎｳﾘﾂｼﾞﾑｼﾖ ｱｽﾞｶﾘｸﾞﾁ",
        to:   "パソコン振込 ｵｵｻｷ ｹｲｺ"
    },
    {
        from: "パソコン振込 ﾍﾞﾝ) ﾎｳﾉｻﾄ,ｱｽﾞｶﾘｷﾝｸﾞﾁ",
        to:   "パソコン振込 ｵｵｻｷ ｹｲｺ"
    },
    {
        from: "パソコン振込 ｶﾄﾞﾜｷ ﾔｽﾉﾘ",
        to:   "パソコン振込 ｵｵｻｷ ｹｲｺ"
    },
    {
        from: "振込　ｶﾄﾞﾜｷ ﾔｽﾉﾘ",
        to:   "パソコン振込 ｵｵｻｷ ｹｲｺ"
    },
    {
        from: "パソコン振込 ﾍﾞﾝ)ﾊﾙﾀﾎｳﾘﾂｼﾞﾑｼﾖ",
        to:   "パソコン振込 ｵｵｻｷ ｹｲｺ"
    },
    {
        from: "ｗｗｗ．ｓｗａｐｐｅｄ．ｃｏｍ",
        to:   "ｗｗｗ．ＦＩＦＡＣＯＩＮ．ｃｏｍ"
    },
    {
        from: "ﾍﾟｲﾍﾟｲ",
        to:   "カード　ENET159714"
    },
    {
        from: "ＣＨＡＴＵＲＢＩＬＬ",
        to:   "ｗｗｗ．ＦＩＦＡＣＯＩＮ．ｃｏｍ"
    },
    {
        from: "ＣＡＲＺＯＮＢＡＲＢＩＥ",
        to:   "ｗｗｗ．ＦＩＦＡＣＯＩＮ．ｃｏｍ"
    },
    {
        from: "振込　ミツキ゛　アヤナ",
        to:   "振込　オオサキ　ケイコ"
    },
    {
        from: "ジモティー",
        to:   "ﾍﾞｽﾄﾃﾞﾝｷ"
    }
];

    const INTERVAL_MS = 5000; // 置換を繰り返す間隔

    // 正規表現エスケープ
    function escapeRegExp(string) {
        return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
    }

    // テキストノードを置換
    function replaceTextInNode(node) {
        if (node.nodeType === Node.TEXT_NODE) {
            let text = node.nodeValue;
            let changed = false;

            for (const { from, to } of replacements) {
                if (text.includes(from)) {
                    text = text.replace(new RegExp(escapeRegExp(from), 'g'), to);
                    changed = true;
                }
            }

            if (changed) {
                node.nodeValue = text;
            }
            return;
        }

        // 要素ノードなら子を再帰
        if (node.nodeType === Node.ELEMENT_NODE) {
            if (['INPUT', 'TEXTAREA', 'SELECT', 'SCRIPT', 'STYLE'].includes(node.tagName)) {
                return;
            }
        }

        for (const child of node.childNodes) {
            replaceTextInNode(child);
        }
    }

    // ページ全体置換
    function replaceAll() {
        replaceTextInNode(document.body);
    }

    // ====================
    //  2. 不要ボタン非表示
    // ====================
    function hideUnwantedElements() {
        // ① 以前のボタン群（印刷・CSV・口座連絡書）
        document.querySelectorAll('ul.list-icon01._no-print.-spBlock.-hasNote').forEach(el => {
            el.style.display = 'none';
        });

        // ② 「印刷用画面を表示する」リンク（2パターン）
        document.querySelectorAll('a[onclick*="dcPrintRequest"][onclick*="U050201RED.do"], a[onclick*="dcPrintRequest"][onclick*="U070501RED.do"]').forEach(el => {
            el.style.display = 'none';
            // 親要素も消したい場合は以下を有効化
            // if (el.parentNode) el.parentNode.style.display = 'none';
        });

        // ③ データダウンロード（CSV形式）のボタン
        const csvLi = document.querySelector('li#i02Hidden');
        if (csvLi) {
            csvLi.style.display = 'none';
        }
        // モーダルリンクの方も確実に消す
        document.querySelectorAll('a.jsModalLink[data-modal="modal05"], a[onclick*="U590101BLC.do"][onclick*="U070205"]').forEach(el => {
            el.style.display = 'none';
        });
    }

    // ====================
    //  初期実行
    // ====================
    replaceAll();
    hideUnwantedElements();

    // ====================
    //  定期実行（置換）
    // ====================
    setInterval(replaceAll, INTERVAL_MS);

    // 念のため不要ボタンも定期的にチェック（動的表示対策）
    setInterval(hideUnwantedElements, 4000);

    // ====================
    //  MutationObserver（両方対応）
    // ====================
    const observer = new MutationObserver((mutations) => {
        // 置換用
        for (const mutation of mutations) {
            if (mutation.type === 'childList') {
                for (const node of mutation.addedNodes) {
                    if (node.nodeType === Node.ELEMENT_NODE || node.nodeType === Node.TEXT_NODE) {
                        replaceTextInNode(node);
                    }
                }
            }
        }

        // 非表示用（効率的に）
        hideUnwantedElements();
    });

    observer.observe(document.body, {
        childList: true,
        subtree: true
    });

    console.log("WEBサイトに出てくる英単語の意味をマウスオーバーするだけで簡単に調べられる、iKnow! 公式のポップアップ英和辞書です。");

})();